/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcUnitUtils;
import gov.nasa.giss.data.nc.NcVariable;
import gov.nasa.giss.time.AbsoluteTime;
import gov.nasa.giss.time.RelativeTime;
import java.lang.invoke.MethodHandles;
import java.text.ParseException;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.DataType;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.Group;
import ucar.nc2.Variable;
import ucar.nc2.dataset.CoordinateSystem;
import ucar.nc2.dataset.CoordinateTransform;
import ucar.nc2.dataset.VariableDS;
import ucar.units.Unit;

public final class NcVarUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

    public static String getStringAttribute(Variable var, String aname) {
        Objects.requireNonNull(var, "Variable cannot be null.");
        Attribute a = var.findAttributeIgnoreCase(aname);
        if (a == null) {
            return null;
        }
        return a.getStringValue();
    }

    public static String getLongName(Variable var) {
        Objects.requireNonNull(var, "Variable cannot be null.");
        Attribute a = var.findAttributeIgnoreCase("long_name");
        if (a == null) {
            a = var.findAttributeIgnoreCase("standard_name");
        }
        if (a == null) {
            a = var.findAttributeIgnoreCase("title");
        }
        if (a == null) {
            return var.getShortName();
        }
        if (a == null) {
            return "";
        }
        return a.getStringValue().replaceAll("\\s*$", "").replace('_', ' ');
    }

    public static boolean isPressure(Variable njvar) {
        Unit uu = NcVarUtils.getUcarUnits(njvar);
        if (uu == null) {
            return false;
        }
        Unit pressure = NcUnitUtils.parse("Pa");
        return uu.isCompatible(pressure);
    }

    public static boolean isHeight(Variable njvar) {
        Unit uu = NcVarUtils.getUcarUnits(njvar);
        if (uu == null) {
            return false;
        }
        Unit distance = NcUnitUtils.parse("m");
        if (!uu.isCompatible(distance)) {
            return false;
        }
        String p = NcVarUtils.getStringAttribute(njvar, "positive");
        return "up".equals(p);
    }

    public static boolean isDepth(Variable njvar) {
        Unit uu = NcVarUtils.getUcarUnits(njvar);
        if (uu == null) {
            return false;
        }
        Unit distance = NcUnitUtils.parse("m");
        if (!uu.isCompatible(distance)) {
            return false;
        }
        String p = NcVarUtils.getStringAttribute(njvar, "positive");
        return "down".equals(p);
    }

    public static String getUnitsString(Variable njvar) {
        if (njvar == null) {
            return "";
        }
        Attribute a = njvar.findAttribute("units");
        if (a == null) {
            a = njvar.findAttribute("Units");
        }
        if (a == null) {
            a = njvar.findAttributeIgnoreCase("unit");
        }
        if (a == null) {
            return "";
        }
        String ss = a.getStringValue();
        if (ss == null) {
            LOGGER.debug("Variable has units attribute that is not a char string.");
            return "";
        }
        return ss.replaceAll("\\s*$", "");
    }

    public static Unit getUcarUnits(Variable njvar) {
        if (njvar == null) {
            return null;
        }
        String unitsStr = NcVarUtils.getUnitsString(njvar);
        try {
            return NcUnitUtils.parse(unitsStr);
        }
        catch (Exception exc) {
            return null;
        }
    }

    public static String[] parseCoordinatesAttribute(Variable njvar) {
        List<Attribute> attribs = njvar.getAttributes();
        if (attribs == null) {
            return null;
        }
        Attribute coordsA = njvar.findAttributeIgnoreCase("coordinates");
        if (coordsA == null) {
            return null;
        }
        String s = coordsA.getStringValue();
        return s.split(" ");
    }

    public static boolean isGriddable(VariableDS varDS) {
        return NcVarUtils.isGriddable1D(varDS) || NcVarUtils.isGriddable2D(varDS);
    }

    public static boolean isGriddable1D(VariableDS varDS) {
        DataType dt = varDS.getDataType();
        if (!dt.isNumeric() || varDS.isScalar()) {
            return false;
        }
        int rank = varDS.getRank();
        int[] shape = varDS.getShape();
        int count = 0;
        for (int i = 0; i < rank; ++i) {
            if (shape[i] == 0) {
                return false;
            }
            if (shape[i] <= 1) continue;
            ++count;
        }
        return count > 0;
    }

    public static boolean isGriddable2D(VariableDS varDS) {
        DataType dt = varDS.getDataType();
        if (!dt.isNumeric() || varDS.isScalar()) {
            return false;
        }
        int rank = varDS.getRank();
        if (rank < 2) {
            return false;
        }
        int count = 0;
        int[] shape = varDS.getShape();
        for (int i = 0; i < rank; ++i) {
            if (shape[i] == 0) {
                return false;
            }
            if (shape[i] <= 1) continue;
            ++count;
        }
        return count > 1;
    }

    public static int findDimensionIndex(NcVariable ncvar, String dimName) {
        return NcVarUtils.findDimensionIndex(ncvar.getNjVariable(), dimName);
    }

    public static int findDimensionIndex(NcVariable ncvar, Dimension dd) {
        return NcVarUtils.findDimensionIndex(ncvar.getNjVariable(), dd);
    }

    public static int findDimensionIndex(VariableDS var, String dimName) {
        if (dimName == null || dimName.isEmpty()) {
            return -1;
        }
        int index = var.findDimensionIndex(dimName);
        if (index > -1) {
            return index;
        }
        List<Dimension> dimensions = var.getDimensions();
        int isize = dimensions.size();
        for (int i = 0; i < isize; ++i) {
            Dimension d = dimensions.get(i);
            boolean matches = dimName.equals(d.makeFullName());
            if (!matches) continue;
            return i;
        }
        return -1;
    }

    public static int findDimensionIndex(VariableDS var, Dimension dd) {
        List<Dimension> dimensions = var.getDimensions();
        int isize = dimensions.size();
        for (int i = 0; i < isize; ++i) {
            Dimension d = dimensions.get(i);
            if (!d.equals(dd)) continue;
            return i;
        }
        return -1;
    }

    public static String getGridMappingName(VariableDS tvarDS) {
        Attribute a = tvarDS.findAttributeIgnoreCase("grid_mapping_name");
        if (a == null) {
            a = tvarDS.findAttributeIgnoreCase("Projection_Name");
        }
        if (a == null) {
            a = tvarDS.findAttributeIgnoreCase("transform_name");
        }
        if (a == null) {
            return null;
        }
        return a.getStringValue();
    }

    public static VariableDS findGridMappingVariable(NcDataset ncd, VariableDS varDS) {
        String tvarName = null;
        String possibleTvarName = null;
        List<CoordinateSystem> csList = varDS.getCoordinateSystems();
        if (csList == null || csList.isEmpty()) {
            LOGGER.trace("cs list null/empty");
        }
        if (csList != null && !csList.isEmpty()) {
            List<CoordinateTransform> ctlist;
            CoordinateSystem cs = csList.get(0);
            List<CoordinateTransform> list = ctlist = cs == null ? null : cs.getCoordinateTransforms();
            if (ctlist != null && ctlist.size() > 0) {
                LOGGER.trace("ctlist size {}", (Object)ctlist.size());
                CoordinateTransform ct = ctlist.get(0);
                String ctname = ct.getName();
                LOGGER.trace("ct name? {}", (Object)ctname);
                VariableDS tvarDS = ncd.getNjVariable(ctname);
                if (tvarDS != null) {
                    LOGGER.trace("transform var found via CS");
                    return tvarDS;
                }
                if (ctlist.size() == 1) {
                    tvarName = ctname;
                } else {
                    possibleTvarName = ctname;
                }
            }
        }
        LOGGER.trace("tvarName {}, possible {}", (Object)tvarName, (Object)possibleTvarName);
        if (tvarName == null) {
            Attribute a = varDS.findAttributeIgnoreCase("grid_mapping");
            LOGGER.trace("grid_mapping att {}", (Object)a);
            if (a == null) {
                a = varDS.findAttributeIgnoreCase("_CoordinateSystems");
                LOGGER.trace("_CoordinateSystems {}", (Object)a);
                if (a == null) {
                    return null;
                }
                VariableDS pcsDS = ncd.getNjVariable(a.getStringValue());
                if (pcsDS == null) {
                    return null;
                }
                a = pcsDS.findAttributeIgnoreCase("_CoordinateTransforms");
                LOGGER.trace("_CoordinateTransforms {}", (Object)a);
            }
            if (a != null) {
                tvarName = a.getStringValue();
            }
        }
        LOGGER.trace("tvarName {}", (Object)tvarName);
        if (tvarName == null && possibleTvarName != null) {
            tvarName = possibleTvarName;
        }
        if (tvarName == null) {
            return null;
        }
        VariableDS mappingVarDS = null;
        Group njgroup = varDS.getParentGroup();
        if (njgroup != null && (mappingVarDS = (VariableDS)njgroup.findVariable(tvarName)) != null) {
            return mappingVarDS;
        }
        mappingVarDS = ncd.getNjVariable(tvarName);
        return mappingVarDS;
    }

    public static boolean isLongitudeVar1D(Variable var) {
        if (var == null || var.getRank() != 1) {
            return false;
        }
        return NcUnitUtils.isDegreesEast(NcVarUtils.getUnitsString(var));
    }

    public static boolean isLatitudeVar1D(Variable var) {
        if (var == null || var.getRank() != 1) {
            return false;
        }
        return NcUnitUtils.isDegreesNorth(NcVarUtils.getUnitsString(var));
    }

    public static boolean isRelativeTimeVar(NcDataset ncd, Variable njvar) {
        String unitsStr = NcVarUtils.getUnitsString(njvar);
        if (unitsStr == null || unitsStr.isEmpty()) {
            return false;
        }
        if (!unitsStr.toLowerCase().contains(" since ")) {
            return false;
        }
        String calendar = null;
        if (njvar.findAttribute("calendar") != null) {
            calendar = njvar.findAttribute("calendar").getStringValue();
        } else if (ncd != null) {
            calendar = ncd.getNjStringAttribute("calendar");
        }
        if (calendar == null) {
            calendar = "gregorian";
        }
        try {
            new RelativeTime(unitsStr, calendar, njvar.getDataType().isFloatingPoint());
            return true;
        }
        catch (Exception exc) {
            return false;
        }
    }

    public static boolean isJulianTimeVar(Variable njvar) {
        String unitsStr = NcVarUtils.getUnitsString(njvar);
        if (unitsStr == null || unitsStr.isEmpty()) {
            return false;
        }
        return "Julian Date".equalsIgnoreCase(unitsStr) || "Julian Day".equalsIgnoreCase(unitsStr) || "True Julian Date".equalsIgnoreCase(unitsStr) || "True Julian Day".equalsIgnoreCase(unitsStr) || "Modified Julian Date".equalsIgnoreCase(unitsStr) || "Modified Julian Day".equalsIgnoreCase(unitsStr);
    }

    public static boolean isAbsoluteTimeVar(Variable v) {
        if (v == null) {
            return false;
        }
        int rank = v.getRank();
        if (rank > 1) {
            return false;
        }
        Attribute unitsA = v.findAttribute("units");
        if (unitsA == null) {
            return false;
        }
        String unitsStr = unitsA.getStringValue();
        if (unitsStr == null) {
            return false;
        }
        try {
            new AbsoluteTime(unitsStr);
            return true;
        }
        catch (Exception exc) {
            return false;
        }
    }

    public static RelativeTime getRelativeTimeHandler(NcDataset dataset, Variable dimVar) throws ParseException {
        String unitsStr = NcVarUtils.getUnitsString(dimVar);
        String calendar = null;
        if (dimVar.findAttribute("calendar") != null) {
            calendar = dimVar.findAttribute("calendar").getStringValue();
        } else if (dataset != null) {
            calendar = dataset.getNjStringAttribute("calendar");
        }
        if (calendar == null) {
            calendar = "gregorian";
        }
        return new RelativeTime(unitsStr, calendar, dimVar.getDataType().isFloatingPoint());
    }

    private NcVarUtils() {
    }
}

