/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.math;

import gov.nasa.giss.math.MathUtils;

public final class Elliptic {
    private static final double TINY_VALUE = 1.0E-25;
    private static final int MAX_ITER = 50;

    public static double ellipticF(double phiRad, double k2) {
        double absPhiRad = Math.abs(phiRad);
        if (absPhiRad > 1.5707963267948966) {
            throw new IllegalArgumentException("phiRad must be in range (0,PI/2).");
        }
        if (k2 < 0.0 || k2 > 1.0) {
            throw new IllegalArgumentException("modulus k^2 must be in range (0,1).");
        }
        double sinPhi = Math.sin(absPhiRad);
        double sin2Phi = sinPhi * sinPhi;
        double x = 1.0 - sin2Phi;
        double y = 1.0 - k2 * sin2Phi;
        double z = 1.0;
        double ellipF = sinPhi * Elliptic.carlsonRF(x, y, 1.0);
        if (phiRad < 0.0) {
            return -ellipF;
        }
        return ellipF;
    }

    public static double ellipticK(double k2) {
        if (k2 < 0.0 || k2 > 1.0) {
            throw new IllegalArgumentException("modulus k^2 must be in range (0,1).");
        }
        double ellipK = Elliptic.carlsonRF(0.0, 1.0 - k2, 1.0);
        return ellipK;
    }

    public static double ellipticE(double phiRad, double k2) {
        double absPhiRad = Math.abs(phiRad);
        if (absPhiRad > 1.5707963267948966) {
            throw new IllegalArgumentException("phiRad must be in range (0,PI/2).");
        }
        if (k2 < 0.0 || k2 > 1.0) {
            throw new IllegalArgumentException("modulus k^2 must be in range (0,1).");
        }
        double sinPhi = Math.sin(absPhiRad);
        double sin2Phi = sinPhi * sinPhi;
        double k2Sin2Phi = k2 * sin2Phi;
        double x = 1.0 - sin2Phi;
        double y = 1.0 - k2Sin2Phi;
        double z = 1.0;
        double ellipE = sinPhi * (Elliptic.carlsonRF(x, y, 1.0) - k2Sin2Phi * Elliptic.carlsonRD(x, y, 1.0) / 3.0);
        if (phiRad < 0.0) {
            return -ellipE;
        }
        return ellipE;
    }

    private static double carlsonRF(double x, double y, double z) throws IllegalArgumentException {
        double zndev;
        double yndev;
        double xndev;
        double mu;
        double UPLIM = 3.0E37;
        double[] dArray = new double[]{x, y, z};
        if (MathUtils.min(dArray) < 0.0) {
            throw new IllegalArgumentException("Negative parameters are not accepted");
        }
        double[] dArray2 = new double[]{x + y, x + z, y + z};
        if (MathUtils.min(dArray2) < 1.0E-25) {
            throw new IllegalArgumentException("At least two parameters must be non-zero");
        }
        double[] dArray3 = new double[]{x, y, z};
        if (MathUtils.max(dArray3) > 3.0E37) {
            throw new IllegalArgumentException("One or more parameters too big");
        }
        double ERRTOL = 0.003;
        double xn = x;
        double yn = y;
        double zn = z;
        int iter = 0;
        do {
            mu = (xn + yn + zn) / 3.0;
            xndev = (mu - xn) / mu;
            yndev = (mu - yn) / mu;
            zndev = (mu - zn) / mu;
            double epsilon = MathUtils.max(Math.abs(xndev), Math.abs(yndev), Math.abs(zndev));
            if (epsilon < 0.003) break;
            double xnroot = Math.sqrt(xn);
            double ynroot = Math.sqrt(yn);
            double znroot = Math.sqrt(zn);
            double lamda = xnroot * (ynroot + znroot) + ynroot * znroot;
            xn = 0.25 * (xn + lamda);
            yn = 0.25 * (yn + lamda);
            zn = 0.25 * (zn + lamda);
        } while (++iter <= 50);
        double e2 = xndev * yndev - zndev * zndev;
        double e3 = xndev * yndev * zndev;
        double C1 = 0.041666666666666664;
        double C2 = 0.06818181818181818;
        double C3 = 0.07142857142857142;
        double ss = 1.0 + (0.041666666666666664 * e2 - 0.1 - 0.06818181818181818 * e3) * e2 + 0.07142857142857142 * e3;
        double rf = ss / Math.sqrt(mu);
        return rf;
    }

    private static double carlsonRD(double x, double y, double z) throws IllegalArgumentException {
        double zndev;
        double yndev;
        double xndev;
        double mu;
        double UPLIM = 4.5E21;
        double[] dArray = new double[]{x, y, z};
        if (MathUtils.min(dArray) < 0.0) {
            throw new IllegalArgumentException("Negative parameters are not accepted");
        }
        if (z == 0.0) {
            throw new IllegalArgumentException("Z parameter must be positive");
        }
        double[] dArray2 = new double[]{x + y, x + z, y + z};
        if (MathUtils.min(dArray2) < 1.0E-25) {
            throw new IllegalArgumentException("At least two parameters must be non-zero");
        }
        double[] dArray3 = new double[]{x, y, z};
        if (MathUtils.max(dArray3) > 4.5E21) {
            throw new IllegalArgumentException("One or more parameters too big");
        }
        double ERRTOL = 0.003;
        double xn = x;
        double yn = y;
        double zn = z;
        double sigma = 0.0;
        double power4 = 1.0;
        int iter = 0;
        do {
            mu = 0.2 * (xn + yn + 3.0 * zn);
            xndev = (mu - xn) / mu;
            yndev = (mu - yn) / mu;
            zndev = (mu - zn) / mu;
            double epsilon = MathUtils.max(Math.abs(xndev), Math.abs(yndev), Math.abs(zndev));
            if (epsilon < 0.003) break;
            double xnroot = Math.sqrt(xn);
            double ynroot = Math.sqrt(yn);
            double znroot = Math.sqrt(zn);
            double lamda = xnroot * (ynroot + znroot) + ynroot * znroot;
            sigma += power4 / (znroot * (zn + lamda));
            power4 = 0.25 * power4;
            xn = 0.25 * (xn + lamda);
            yn = 0.25 * (yn + lamda);
            zn = 0.25 * (zn + lamda);
        } while (++iter <= 50);
        double ea = xndev * yndev;
        double eb = zndev * zndev;
        double ec = ea - eb;
        double ed = ea - 6.0 * eb;
        double ef = ed + ec + ec;
        double C1 = 0.21428571428571427;
        double C2 = 0.16666666666666666;
        double C3 = 0.4090909090909091;
        double C4 = 0.11538461538461539;
        double s1 = ed * (-0.21428571428571427 + 0.10227272727272728 * ed - 0.17307692307692307 * zndev * ef);
        double s2 = zndev * (0.16666666666666666 * ef + zndev * (-0.4090909090909091 * ec + zndev * 0.11538461538461539 * ea));
        double rd = 3.0 * sigma + power4 * (1.0 + s1 + s2) / (mu * Math.sqrt(mu));
        return rd;
    }

    public static double[] jacobiSnCnDn(double u, double k2) {
        if (k2 < 0.0 || k2 > 1.0) {
            throw new IllegalArgumentException("k^2 must be in range [0,1]");
        }
        double kc2 = 1.0 - k2;
        if (kc2 == 0.0) {
            double cn;
            double dn = cn = 1.0 / Math.cosh(u);
            double sn = Math.tanh(u);
            return new double[]{sn, cn, dn};
        }
        double ca = 3.0E-4;
        double kcc = kc2;
        double uu = u;
        boolean bo = kcc < 0.0;
        double d = 1.0;
        if (bo) {
            d = 1.0 - kcc;
            kcc /= -1.0 / d;
            d = Math.sqrt(d);
            uu *= d;
        }
        double a = 1.0;
        double dn = 1.0;
        double[] em = new double[14];
        double[] en = new double[14];
        int j = 1;
        double c = 1.0;
        for (int i = 1; i <= 13; ++i) {
            j = i;
            em[i] = a;
            en[i] = kcc = Math.sqrt(kcc);
            c = 0.5 * (a + kcc);
            if (Math.abs(a - kcc) <= 3.0E-4 * a) break;
            kcc *= a;
            a = c;
        }
        double sn = Math.sin(uu *= c);
        double cn = Math.cos(uu);
        if (sn != 0.0) {
            a = cn / sn;
            c *= a;
            for (int ii = j; ii >= 1; --ii) {
                double b = em[ii];
                dn = (en[ii] + (a *= c)) / (b + a);
                a = (c *= dn) / b;
            }
            a = 1.0 / Math.sqrt(c * c + 1.0);
            sn = sn >= 0.0 ? a : -a;
            cn = c * sn;
        }
        if (bo) {
            a = dn;
            dn = cn;
            cn = a;
            sn /= d;
        }
        return new double[]{sn, cn, dn};
    }

    private Elliptic() {
    }
}

