/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.panoply.action.task;

import gov.nasa.giss.data.nc.NcArray;
import gov.nasa.giss.data.nc.NcDimension;
import gov.nasa.giss.img.ImageUtils;
import gov.nasa.giss.panoply.data.PanData;
import gov.nasa.giss.panoply.plot.PanPlot;
import gov.nasa.giss.panoply.plot.PanPlotMeta;
import gov.nasa.giss.panoply.plotui.PanPlotFrame;
import gov.nasa.giss.panoply.plotui.PanPlotHolder;
import gov.nasa.giss.panoply.util.PanAnimationFormat;
import gov.nasa.giss.panoply.util.PanAnimationOption;
import gov.nasa.giss.panoply.util.PanConstants;
import gov.nasa.giss.text.PrintfFormat;
import gov.nasa.giss.ui.GuiUtils;
import gov.nasa.giss.util.task.Task;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.Hashtable;
import org.jcodec.api.awt.AWTSequenceEncoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PanExportAnimationTask
extends Task {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final PrintfFormat PFORMAT_5D = new PrintfFormat("%05d");
    private final PanPlotFrame pframe_;
    private final File file_;
    private final Hashtable<PanAnimationOption, Object> options_;
    private final NcDimension[] dimensions_;
    private final PanAnimationFormat format_;
    private PanPlotMeta pmeta_;
    private PanData pdata_;
    private int[] sliceDims_;
    private int[] sliceIndices_;
    private int[] dimlengths_;
    private final boolean showSubtitle_;
    private final boolean subBounds_;
    private File oldFile_;
    private BufferedImage bimage_;

    public PanExportAnimationTask(File file, PanPlotFrame pframe, NcDimension[] dimensions, Hashtable<PanAnimationOption, Object> options) {
        super("Creating plot animation");
        this.file_ = file;
        this.pframe_ = pframe;
        this.dimensions_ = dimensions;
        this.options_ = options;
        this.format_ = (PanAnimationFormat)((Object)options.get((Object)PanAnimationOption.FORMAT));
        this.showSubtitle_ = (Boolean)options.get((Object)PanAnimationOption.SUBTITLE_STEP);
        this.subBounds_ = (Boolean)options.get((Object)PanAnimationOption.SUBTITLE_BOUNDS);
    }

    @Override
    protected Object beginTask() {
        if (this.isCancelled()) {
            return null;
        }
        if (!this.pframe_.isVisible()) {
            LOGGER.debug("Frame no longer visible; canceling animation.");
            this.cancel(true);
            return null;
        }
        String oldSubtitle = this.pframe_.getMeta().getString("subtitle.text");
        boolean successful = false;
        try {
            if (this.file_.exists()) {
                this.oldFile_ = new File(this.file_.getParent(), this.file_.getName() + "_old");
                this.file_.renameTo(this.oldFile_);
            }
        }
        catch (Exception exc) {
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            GuiUtils.showSlowMessage(this.pframe_, "Animation Error", "Could not save animation to a file. " + exc.toString());
        }
        try {
            successful = this.writeAnimation();
        }
        catch (NullPointerException exc) {
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            GuiUtils.showSlowMessage(this.pframe_, "Animation Error", "Animation terminated by plot window closing.");
        }
        catch (Exception exc) {
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            GuiUtils.showSlowMessage(this.pframe_, "Animation Error", "Could not save animation to a file. " + exc.toString());
        }
        if (successful) {
            if (this.oldFile_ != null && this.oldFile_.exists()) {
                this.oldFile_.delete();
            }
        } else {
            if (this.file_.exists()) {
                this.file_.delete();
            }
            if (this.oldFile_ != null && this.oldFile_.exists()) {
                this.oldFile_.renameTo(this.file_);
            }
        }
        this.pframe_.getMeta().setString("subtitle.text", oldSubtitle);
        return null;
    }

    @Override
    protected void finishTask() {
        this.pframe_.setAnimationInProgress(false);
        this.pframe_.repaint();
    }

    private boolean writeAnimation() throws InterruptedException, IOException {
        if (this.isCancelled()) {
            return false;
        }
        if (this.format_ == null) {
            throw new RuntimeException("Null format");
        }
        this.pdata_ = this.pframe_.getData();
        this.pmeta_ = this.pframe_.getPlot().getMeta();
        Dimension psize = (Dimension)this.pmeta_.getLayout().getParam("size.pixels");
        int frameCount = 1;
        this.sliceDims_ = new int[]{-9999, -9999};
        this.sliceIndices_ = new int[]{-9999, -9999};
        this.dimlengths_ = new int[]{-1, -1};
        for (int i = 0; i < 2; ++i) {
            if (this.dimensions_[i] == null) continue;
            NcArray array = this.pframe_.getData().getArrays()[i];
            this.sliceDims_[i] = array.getDimensionId(this.dimensions_[i]);
            this.sliceIndices_[i] = this.pdata_.getSlice(i, this.sliceDims_[i]);
            this.dimlengths_[i] = this.dimensions_[i].getLength();
            frameCount = Math.max(frameCount, this.dimlengths_[i]);
        }
        this.bimage_ = new BufferedImage(psize.width, psize.height, 1);
        boolean success = true;
        switch (this.format_) {
            case MP4: {
                success = this.writeVideo();
                break;
            }
            case PNG: 
            case JPEG: 
            case TIFF: {
                success = this.writeStills();
                break;
            }
            default: {
                throw new RuntimeException("Unknown animation format.");
            }
        }
        this.cleanup();
        return success;
    }

    private boolean writeVideo() throws IOException {
        int frameRate = (Integer)this.options_.get((Object)PanAnimationOption.FRAME_RATE);
        int startIndex = (Integer)this.options_.get((Object)PanAnimationOption.START_INDEX);
        int stopIndex = (Integer)this.options_.get((Object)PanAnimationOption.STOP_INDEX);
        int stride = (Integer)this.options_.get((Object)PanAnimationOption.STRIDE);
        AWTSequenceEncoder encoder = AWTSequenceEncoder.createSequenceEncoder((File)this.file_, (int)frameRate);
        for (int i = startIndex - 1; i < stopIndex; i += stride) {
            if (this.isCancelled()) {
                this.cleanup();
                return false;
            }
            boolean good = this.writeFrame(this.bimage_, i);
            if (good) {
                try {
                    encoder.encodeImage(this.bimage_);
                    continue;
                }
                catch (Exception exc) {
                    if (!this.isCancelled()) {
                        LOGGER.error("Failed to encode image frame due to exception. Canceling.");
                        if (LOGGER.isTraceEnabled()) {
                            exc.printStackTrace();
                        }
                        this.cancel(true);
                    }
                    this.cleanup();
                    return false;
                }
            }
            if (!this.isCancelled()) {
                LOGGER.error("Failed to generate video frame. Canceling.");
                this.cancel(true);
                this.cleanup();
            }
            return false;
        }
        encoder.finish();
        return true;
    }

    private boolean writeStills() throws IOException {
        String fname = this.file_.getName();
        int lastDot = fname.lastIndexOf(46);
        String froot = fname.substring(0, lastDot);
        String fext = fname.substring(lastDot);
        int quality = (Integer)this.options_.get((Object)PanAnimationOption.QUALITY);
        float qualityF = quality < 95 ? 0.01f * (float)quality : 1.0f;
        int startIndex = (Integer)this.options_.get((Object)PanAnimationOption.START_INDEX);
        int stopIndex = (Integer)this.options_.get((Object)PanAnimationOption.STOP_INDEX);
        int stride = (Integer)this.options_.get((Object)PanAnimationOption.STRIDE);
        for (int i = startIndex - 1; i < stopIndex; i += stride) {
            if (this.isCancelled()) {
                this.cleanup();
                return false;
            }
            boolean good = this.writeFrame(this.bimage_, i);
            if (!good) {
                LOGGER.warn("Failed to write still frame. Canceling.");
                this.cancel(true);
                this.cleanup();
                return false;
            }
            if (this.format_ != PanAnimationFormat.PNG && this.format_ != PanAnimationFormat.JPEG && this.format_ != PanAnimationFormat.TIFF) continue;
            fname = froot + "_" + PFORMAT_5D.sprintf((Object)(i + 1)) + fext;
            File f = new File(this.file_.getParent(), fname);
            if (this.format_ == PanAnimationFormat.PNG) {
                ImageUtils.saveAsPng(f, this.bimage_);
                continue;
            }
            if (this.format_ == PanAnimationFormat.JPEG) {
                ImageUtils.saveAsJpeg(f, this.bimage_, qualityF);
                continue;
            }
            if (this.format_ != PanAnimationFormat.TIFF) continue;
            ImageUtils.saveAsTiff(f, this.bimage_, PanConstants.SOFTWARE);
        }
        return true;
    }

    private boolean writeFrame(BufferedImage bimage, int findex) {
        PanPlot plot;
        String fbds;
        String fval;
        boolean fromBounds;
        if (!this.pframe_.isVisible()) {
            LOGGER.debug("Plot window closed; canceling animation.");
            this.cancel(true);
            return false;
        }
        String s0 = null;
        String s1 = null;
        if (this.dimlengths_[0] > findex) {
            this.pdata_.setSlice(0, this.sliceDims_[0], findex);
            fromBounds = this.subBounds_ && this.dimensions_[0].hasBounds();
            fval = this.dimensions_[0].formattedValueAt(findex);
            fbds = this.dimensions_[0].formattedBoundsAt(findex);
            s0 = this.dimensions_[0].getLongName() + ": " + (fromBounds ? fbds : fval);
        }
        if (this.dimlengths_[1] > findex) {
            this.pdata_.setSlice(1, this.sliceDims_[1], findex);
            fromBounds = this.subBounds_ && this.dimensions_[1].hasBounds();
            fval = this.dimensions_[1].formattedValueAt(findex);
            fbds = this.dimensions_[1].formattedBoundsAt(findex);
            s1 = this.dimensions_[1].getLongName() + ": " + (fromBounds ? fbds : fval);
        }
        if (this.showSubtitle_) {
            if (s0 != null && s1 != null) {
                this.pmeta_.setString("subtitle.text", s0 + " : " + s1);
            } else if (s0 != null) {
                this.pmeta_.setString("subtitle.text", s0);
            } else if (s1 != null) {
                this.pmeta_.setString("subtitle.text", s1);
            } else {
                this.pmeta_.setString("subtitle.text", "");
            }
        }
        if ((plot = this.pframe_.getPlot()) == null) {
            if (!this.isCancelled()) {
                LOGGER.warn("Frame's plot is null. Skipping frame write.");
            }
            return false;
        }
        plot.paint(bimage);
        PanPlotHolder pph = this.pframe_.getPlotHolder();
        if (pph != null) {
            EventQueue.invokeLater(() -> pph.repaint());
        }
        return true;
    }

    private final void cleanup() {
        PanPlot plot;
        if (this.sliceDims_[0] > -9999) {
            this.pdata_.setSlice(0, this.sliceDims_[0], this.sliceIndices_[0]);
        }
        if (this.sliceDims_[1] > -9999) {
            this.pdata_.setSlice(1, this.sliceDims_[1], this.sliceIndices_[1]);
        }
        if ((plot = this.pframe_.getPlot()) == null) {
            if (!this.isCancelled()) {
                LOGGER.warn("Frame says plot is null. Terminating clean-up.");
            }
            return;
        }
        plot.paint(this.bimage_);
        EventQueue.invokeLater(() -> {
            if (this.pframe_ == null) {
                return;
            }
            this.pframe_.getPlotHolder().repaint();
        });
    }
}

