/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import gov.nasa.giss.data.nc.NcAxisType;
import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcException;
import gov.nasa.giss.data.nc.NcVarUtils;
import java.lang.invoke.MethodHandles;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.nc2.Attribute;
import ucar.nc2.Variable;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.VariableDS;
import ucar.units.DerivedUnit;
import ucar.units.Unit;

public class NcAxis {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final String name_;
    private final String sname_;
    private CoordinateAxis njaxis_;
    protected Variable njaxisvar_;
    private AxisType njType_;
    protected NcDataset dataset_;
    private int numVals_;
    private double[] values_;
    private double[][] bounds_;
    private boolean wraparound_;
    private int redundant_;
    private String units_;

    public NcAxis(NcAxisType type, String name, int length) {
        LOGGER.trace("constructor 1 for {}", (Object)name);
        this.name_ = name;
        this.sname_ = name;
        this.numVals_ = length;
        switch (type) {
            case GEOX: 
            case X: {
                this.njType_ = AxisType.GeoX;
                break;
            }
            case GEOY: 
            case Y: {
                this.njType_ = AxisType.GeoY;
                break;
            }
            case VERT: {
                this.njType_ = AxisType.GeoZ;
                break;
            }
            case TIME: {
                this.njType_ = AxisType.Time;
                break;
            }
            case SPECTRAL: {
                this.njType_ = AxisType.Spectral;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to create axis of type " + (Object)((Object)type));
            }
        }
        this.values_ = new double[this.numVals_];
        this.bounds_ = new double[this.numVals_][2];
        for (int i = 0; i < this.numVals_; ++i) {
            double val;
            this.values_[i] = val = (double)i;
            this.bounds_[i][0] = val - 0.5;
            this.bounds_[i][1] = val + 0.5;
        }
    }

    public NcAxis(NcDataset dataset, CoordinateAxis njaxis) {
        LOGGER.trace("constructor 3");
        Objects.requireNonNull(dataset, "Null dataset argument.");
        Objects.requireNonNull(njaxis, "Null axis argument.");
        int rank = njaxis.getRank();
        if (rank == 0) {
            throw new IllegalArgumentException("Axis var is a scalar.");
        }
        if (rank > 1) {
            throw new IllegalArgumentException("Axis var is > 1D.");
        }
        this.dataset_ = dataset;
        this.njaxis_ = njaxis;
        this.njaxisvar_ = njaxis;
        this.njType_ = this.njaxis_.getAxisType();
        this.name_ = this.njaxis_.getFullName();
        this.sname_ = this.njaxis_.getShortName();
        this.numVals_ = this.njaxis_.getShape()[0];
        this.values_ = new double[this.numVals_];
        Array vArray = null;
        try {
            vArray = njaxis.read();
        }
        catch (Exception exc) {
            LOGGER.error("Could not read axis {} -- {}", (Object)this.name_, (Object)exc.toString());
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException(String.format("Could not read axis %s", this.name_));
        }
        this.initValues(vArray);
    }

    public NcAxis(NcAxisType type, NcDataset dataset, Variable njvar) {
        LOGGER.trace("constructor 4");
        Objects.requireNonNull(type, "Null axis type.");
        Objects.requireNonNull(dataset, "Null dataset argument.");
        Objects.requireNonNull(njvar, "Null axis variable argument.");
        this.sname_ = njvar.getShortName();
        LOGGER.trace(":: {}, type {}", (Object)this.sname_, (Object)type);
        int rank = njvar.getRank();
        if (rank == 0) {
            throw new IllegalArgumentException(String.format("Axis var $s is a scalar.", this.sname_));
        }
        if (rank > 1) {
            throw new IllegalArgumentException(String.format("Axis var $s is > 1D.", this.sname_));
        }
        switch (type) {
            case GEOX: {
                this.njType_ = AxisType.GeoX;
                break;
            }
            case GEOY: {
                this.njType_ = AxisType.GeoY;
                break;
            }
            case VERT: {
                if (NcVarUtils.isPressure(njvar)) {
                    this.njType_ = AxisType.Pressure;
                    break;
                }
                if (NcVarUtils.isHeight(njvar)) {
                    this.njType_ = AxisType.Height;
                    break;
                }
                this.njType_ = AxisType.GeoZ;
                break;
            }
            case LON: {
                this.njType_ = AxisType.Lon;
                break;
            }
            case LAT: {
                this.njType_ = AxisType.Lat;
                break;
            }
            case TIME: {
                this.njType_ = AxisType.Time;
                break;
            }
            case SPECTRAL: {
                this.njType_ = AxisType.Spectral;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to create axis of type " + (Object)((Object)type));
            }
        }
        this.dataset_ = dataset;
        this.njaxis_ = null;
        this.njaxisvar_ = njvar;
        this.name_ = njvar.getFullName();
        this.numVals_ = njvar.getShape()[0];
        this.values_ = new double[this.numVals_];
        try {
            this.initValues(njvar.read());
        }
        catch (Exception exc) {
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException(String.format("Could not initialize values for %s -- %s", this.sname_, exc.toString()));
        }
    }

    public NcAxis(NcAxisType type, String name, double[] values) {
        LOGGER.trace("constructor 2 for {}", (Object)name);
        this.name_ = name;
        this.sname_ = name;
        this.numVals_ = values.length;
        switch (type) {
            case GEOX: {
                this.njType_ = AxisType.GeoX;
                break;
            }
            case VERT: {
                this.njType_ = AxisType.GeoZ;
                break;
            }
            case LON: {
                this.njType_ = AxisType.Lon;
                break;
            }
            case LAT: {
                this.njType_ = AxisType.Lat;
                break;
            }
            case TIME: {
                this.njType_ = AxisType.Time;
                break;
            }
            case SPECTRAL: {
                this.njType_ = AxisType.Spectral;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to create axis of type " + (Object)((Object)type));
            }
        }
        int last = this.numVals_ - 1;
        this.values_ = new double[this.numVals_];
        this.bounds_ = new double[this.numVals_][2];
        if (this.numVals_ == 1) {
            this.values_[0] = values[0];
            if (this.njType_ == AxisType.Lon) {
                this.bounds_[0][0] = values[0] - 180.0;
                this.bounds_[0][1] = values[0] + 180.0;
            } else {
                this.bounds_[0][0] = values[0];
                this.bounds_[0][1] = values[0];
            }
        } else {
            for (int i = 0; i < this.numVals_; ++i) {
                this.values_[i] = values[i];
                if (i == 0) {
                    this.bounds_[0][0] = values[0];
                    this.bounds_[i][1] = 0.5 * (values[0] + values[1]);
                    continue;
                }
                if (i == last) {
                    this.bounds_[i][0] = 0.5 * (values[last - 1] + values[last]);
                    this.bounds_[i][1] = values[last];
                    continue;
                }
                this.bounds_[i][0] = 0.5 * (values[i - 1] + values[i]);
                this.bounds_[i][1] = 0.5 * (values[i] + values[i + 1]);
            }
            if (this.njType_ == AxisType.Lon) {
                double gap1 = values[1] - values[0];
                double gapX = values[last] - values[last - 1];
                this.bounds_[0][0] = values[0] - gap1;
                this.bounds_[last][1] = values[last] + gapX;
            }
        }
    }

    private final void initValues(Array vArray) {
        int j;
        for (j = 0; j < this.numVals_; ++j) {
            double value = vArray.getDouble(vArray.getIndex().set(j));
            if (Double.isNaN(value)) {
                LOGGER.error("NaN found :: {}, {}", (Object)j, (Object)value);
                throw new NcException("Axis includes NaN value(s).");
            }
            if (j > 0 && value == this.values_[j - 1]) {
                LOGGER.error("Not unique :: {}, {} :: {}, {}", j - 1, this.values_[j - 1], j, value);
                throw new NcException("Axis " + this.sname_ + " does not have unique values; found consecutive " + value + " values.");
            }
            this.values_[j] = value;
        }
        if (this.njType_ == AxisType.Lon) {
            for (j = 1; j < this.numVals_; ++j) {
                if (!(this.values_[j] < this.values_[j - 1])) continue;
                int n = j;
                this.values_[n] = this.values_[n] + 360.0;
            }
        }
        this.bounds_ = new double[this.numVals_][2];
        boolean gotBounds = this.readBounds();
        if (!gotBounds) {
            gotBounds = this.readEdges();
        }
        if (!gotBounds) {
            gotBounds = this.calculateBounds();
        }
        if (!gotBounds) {
            throw new NcException("Unable to extract axis value bounds.");
        }
        this.fixBounds();
    }

    private boolean readBounds() {
        Array bArray;
        Attribute ba;
        if (this.njaxisvar_ == null) {
            return false;
        }
        String boundsName = null;
        if (this.njaxis_ != null) {
            boundsName = this.njaxis_.getBoundaryRef();
        }
        if (boundsName == null && (ba = this.njaxisvar_.findAttribute("bounds")) != null) {
            boundsName = ba.getStringValue();
        }
        if (boundsName == null && (ba = this.njaxisvar_.findAttribute("climatology")) != null) {
            boundsName = ba.getStringValue();
        }
        if (boundsName == null) {
            return false;
        }
        VariableDS boundsVar = this.dataset_.getNjVariable(boundsName);
        if (boundsVar == null) {
            return false;
        }
        try {
            bArray = boundsVar.read();
        }
        catch (Exception exc) {
            throw new NcException(String.format("Could not read axis bounds $s", boundsName));
        }
        int[] bshape = boundsVar.getShape();
        if (bshape[0] == 2 && bshape[1] > 2) {
            bArray = bArray.transpose(0, 1);
        }
        try {
            if (this.numVals_ > 1) {
                double val0 = this.values_[0];
                double val1 = this.values_[1];
                double b10 = bArray.getDouble(bArray.getIndex().set(1, 0));
                double b11 = bArray.getDouble(bArray.getIndex().set(1, 1));
                if (Double.isNaN(b10)) {
                    b10 = 0.0;
                }
                if (Double.isNaN(b11)) {
                    b11 = 0.0;
                }
                if (b10 < val0 && b10 < val1 || b10 > val0 && b10 > val1) {
                    bArray = bArray.flip(1);
                }
            }
            for (int i = 0; i < this.numVals_; ++i) {
                double b0 = bArray.getDouble(bArray.getIndex().set(i, 0));
                double b1 = bArray.getDouble(bArray.getIndex().set(i, 1));
                if (Double.isNaN(b0)) {
                    b0 = 0.0;
                }
                if (Double.isNaN(b1)) {
                    b1 = 0.0;
                }
                if (this.njType_ == AxisType.Lon) {
                    while (b0 > this.values_[i]) {
                        b0 -= 360.0;
                    }
                    while (b1 < this.values_[i]) {
                        b1 += 360.0;
                    }
                    while (this.values_[i] - b0 > 360.0) {
                        b0 += 360.0;
                    }
                    while (b1 - this.values_[i] > 360.0) {
                        b1 -= 360.0;
                    }
                }
                this.bounds_[i][0] = b0;
                this.bounds_[i][1] = b1;
            }
            return true;
        }
        catch (Exception exc) {
            LOGGER.warn("Failed to read axis bound values from dataset.");
            LOGGER.warn("Exception reported: {}", (Object)exc.toString());
            return false;
        }
    }

    private boolean readEdges() {
        Array eArray;
        if (this.njaxisvar_ == null) {
            return false;
        }
        Attribute edgesA = this.njaxisvar_.findAttribute("edges");
        if (edgesA == null) {
            return false;
        }
        String edgesName = edgesA.getStringValue();
        if (edgesName == null) {
            return false;
        }
        VariableDS edgesVar = this.dataset_.getNjVariable(edgesName);
        try {
            eArray = edgesVar.read();
        }
        catch (Exception exc) {
            throw new NcException(String.format("Could not read axis edges %s", edgesName));
        }
        for (int i = 0; i < this.numVals_; ++i) {
            double b0 = eArray.getDouble(eArray.getIndex().set(i));
            double b1 = eArray.getDouble(eArray.getIndex().set(i + 1));
            if (Double.isNaN(b0)) {
                b0 = 0.0;
            }
            if (Double.isNaN(b1)) {
                b1 = 0.0;
            }
            if (this.njType_ == AxisType.Lon) {
                while (b0 > this.values_[i]) {
                    b0 -= 360.0;
                }
                while (b1 < this.values_[i]) {
                    b1 += 360.0;
                }
                while (this.values_[i] - b0 > 360.0) {
                    b0 += 360.0;
                }
                while (b1 - this.values_[i] > 360.0) {
                    b1 -= 360.0;
                }
            }
            this.bounds_[i][0] = b0;
            this.bounds_[i][1] = b1;
        }
        return true;
    }

    private final boolean calculateBounds() {
        if (this.numVals_ > 1) {
            int last = this.numVals_ - 1;
            for (int i = 0; i < last; ++i) {
                this.bounds_[i][1] = 0.5 * (this.values_[i] + this.values_[i + 1]);
                this.bounds_[i + 1][0] = this.bounds_[i][1];
            }
            this.bounds_[0][0] = 2.0 * this.values_[0] - this.bounds_[0][1];
            this.bounds_[last][1] = 2.0 * this.values_[last] - this.bounds_[last][0];
            if (this.isPressure()) {
                if (this.bounds_[0][0] <= 0.0) {
                    this.bounds_[0][0] = this.values_[0];
                }
                if (this.bounds_[last][1] <= 0.0) {
                    this.bounds_[last][1] = this.values_[last];
                }
            }
        } else if (this.njType_ == AxisType.Lon) {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        } else if (this.njType_ == AxisType.Lat) {
            this.bounds_[0][0] = this.values_[0] + 5.0;
            this.bounds_[0][1] = this.values_[0] - 5.0;
        } else {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        }
        return true;
    }

    private final void fixBounds() {
        if (this.njType_ != AxisType.Lon) {
            return;
        }
        int last = this.numVals_ - 1;
        if (this.numVals_ > 2) {
            if (this.values_[last - 1] >= this.values_[0] + 360.0) {
                this.redundant_ = 2;
            } else if (this.values_[last] >= this.values_[0] + 360.0) {
                this.redundant_ = 1;
            }
            if (this.redundant_ > 0) {
                this.numVals_ -= this.redundant_;
                last = this.numVals_ - 1;
                double[] newvals = new double[this.numVals_];
                double[][] newbounds = new double[this.numVals_][2];
                for (int i = 0; i < this.numVals_; ++i) {
                    newvals[i] = this.values_[i];
                    newbounds[i][0] = this.bounds_[i][0];
                    newbounds[i][1] = this.bounds_[i][1];
                }
                this.values_ = newvals;
                this.bounds_ = newbounds;
            }
        }
        if (this.numVals_ > 1) {
            double delta = this.values_[last] - this.values_[last - 1];
            double overlap = this.bounds_[last][1] - (this.bounds_[0][0] + 360.0);
            boolean bl = this.wraparound_ = overlap >= 0.0 || Math.abs(overlap / delta) < 0.2;
            if (overlap > 0.0) {
                double[] dArray = this.bounds_[0];
                dArray[0] = dArray[0] + (overlap *= 0.5);
                double[] dArray2 = this.bounds_[last];
                dArray2[1] = dArray2[1] - overlap;
            }
        }
        for (int i = 0; i < this.numVals_; ++i) {
            while (this.bounds_[i][0] > this.values_[i]) {
                double[] dArray = this.bounds_[i];
                dArray[0] = dArray[0] - 360.0;
            }
            while (this.values_[i] - this.bounds_[i][0] >= 360.0) {
                double[] dArray = this.bounds_[i];
                dArray[0] = dArray[0] + 360.0;
            }
            while (this.bounds_[i][1] < this.values_[i]) {
                double[] dArray = this.bounds_[i];
                dArray[1] = dArray[1] + 360.0;
            }
            while (this.bounds_[i][1] - this.values_[i] >= 360.0) {
                double[] dArray = this.bounds_[i];
                dArray[1] = dArray[1] - 360.0;
            }
        }
    }

    public String getName() {
        return this.name_;
    }

    public CoordinateAxis getNjAxis() {
        return this.njaxis_;
    }

    public boolean hasPositive() {
        if (this.njType_ == AxisType.Pressure) {
            return true;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height || this.njType_ == AxisType.Pressure) {
            if (this.njaxis_ != null) {
                String p = this.njaxis_.getPositive();
                return p != null;
            }
            if (this.njaxisvar_ != null) {
                String p = NcVarUtils.getStringAttribute(this.njaxisvar_, "positive");
                return p != null;
            }
        }
        return false;
    }

    public boolean isPositiveUp() {
        if (this.njType_ == AxisType.Pressure) {
            return false;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height) {
            if (this.njaxis_ != null) {
                String p = this.njaxis_.getPositive();
                return p != null && p.equals("up");
            }
            if (this.njaxisvar_ != null) {
                String p = NcVarUtils.getStringAttribute(this.njaxisvar_, "positive");
                return "up".equalsIgnoreCase(p);
            }
        }
        return false;
    }

    public boolean isPositiveDown() {
        LOGGER.trace("");
        if (this.njType_ == AxisType.Pressure) {
            LOGGER.trace("PRESSURE");
            return true;
        }
        if (this.njType_ == AxisType.GeoZ || this.njType_ == AxisType.Height) {
            LOGGER.trace("Z/HEIGHT");
            if (this.njaxis_ != null) {
                String p = this.njaxis_.getPositive();
                LOGGER.trace("p? {}", (Object)p);
                return p != null && p.equals("down");
            }
            if (this.njaxisvar_ != null) {
                String p = NcVarUtils.getStringAttribute(this.njaxisvar_, "positive");
                LOGGER.trace("a? {}", (Object)p);
                return p != null && p.equals("down");
            }
        }
        return false;
    }

    public boolean isPressure() {
        if (this.njType_ == AxisType.Pressure) {
            return true;
        }
        Unit uu = this.getUcarUnits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        return du.toString().equals("Pa");
    }

    public boolean isHeight() {
        if (this.njType_ == AxisType.Height) {
            return true;
        }
        Unit uu = this.getUcarUnits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        return du.toString().equals("m");
    }

    public boolean isDepth() {
        Unit uu = this.getUcarUnits();
        if (uu == null) {
            return false;
        }
        DerivedUnit du = uu.getDerivedUnit();
        if (du == null) {
            return false;
        }
        boolean isDist = du.toString().equals("m");
        if (!isDist) {
            return false;
        }
        return this.isPositiveDown();
    }

    public boolean isPatternTime() {
        return false;
    }

    public int getLength() {
        return this.values_.length;
    }

    public double[] getValues() {
        return this.values_;
    }

    public double[][] getBounds() {
        return this.bounds_;
    }

    public String getLongName() {
        if (this.njaxisvar_ != null) {
            return NcVarUtils.getLongName(this.njaxisvar_);
        }
        if (this.name_ != null && !this.name_.isEmpty()) {
            return this.name_;
        }
        if (this.njType_ == AxisType.GeoX || this.njType_ == AxisType.GeoY) {
            return "Axis " + this.getLength();
        }
        return "";
    }

    public String getUnitsString() {
        if (this.units_ == null) {
            this.units_ = this.njaxisvar_ == null ? "" : NcVarUtils.getUnitsString(this.njaxisvar_);
        }
        return this.units_;
    }

    public Unit getUcarUnits() {
        return NcVarUtils.getUcarUnits(this.njaxisvar_);
    }

    public double valueAt(int index) {
        return this.values_[index];
    }

    public double[] boundsAt(int index) {
        return new double[]{this.bounds_[index][0], this.bounds_[index][1]};
    }

    public double boundsAt(int index1, int index2) {
        return this.bounds_[index1][index2];
    }

    public int findNearestIndex(double value) {
        double xvalue;
        if (this.njType_ == AxisType.Lat && (value < -90.0 || value > 90.0)) {
            return -1;
        }
        if (this.njType_ == AxisType.Lon) {
            int last = this.numVals_ - 1;
            for (xvalue = value; xvalue < this.bounds_[0][0] && xvalue < this.bounds_[last][1]; xvalue += 360.0) {
            }
            while (xvalue > this.bounds_[0][0] && xvalue > this.bounds_[last][1]) {
                xvalue -= 360.0;
            }
        }
        for (int i = 0; i < this.numVals_; ++i) {
            if (xvalue < this.bounds_[i][0] && xvalue < this.bounds_[i][1] || xvalue > this.bounds_[i][0] && xvalue > this.bounds_[i][1]) continue;
            return i;
        }
        return -1;
    }

    public boolean isWraparound() {
        return this.wraparound_;
    }

    public boolean isCompatible(NcAxis pa) {
        Objects.requireNonNull(pa, "Null axis.");
        return this.isCompatible(pa.getUcarUnits());
    }

    public boolean isCompatible(Unit u2) {
        Unit u1 = this.getUcarUnits();
        if (u1 == null && u2 == null) {
            return true;
        }
        if (u1 == null || u2 == null) {
            return false;
        }
        return u2.isCompatible(u1);
    }

    public boolean equals(Object o) {
        if (o == null) {
            return false;
        }
        if (o == this) {
            return true;
        }
        if (!(o instanceof NcAxis)) {
            return false;
        }
        NcAxis a2 = (NcAxis)o;
        if (a2.getLength() != this.getLength()) {
            return false;
        }
        if (!a2.getUnitsString().equals(this.getUnitsString())) {
            return false;
        }
        double[] a2Values = a2.getValues();
        double[][] a2Bounds = a2.getBounds();
        for (int i = 0; i < this.values_.length; ++i) {
            if (a2Values[i] != this.values_[i]) {
                return false;
            }
            if (a2Bounds[i][0] != this.bounds_[i][0]) {
                return false;
            }
            if (a2Bounds[i][1] == this.bounds_[i][1]) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        if (this.njaxis_ == null) {
            return this.values_.hashCode();
        }
        return this.njaxis_.hashCode();
    }
}

