import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;

public class DateUtils {
	
	private final int datePosition;
	private final Sensor.Duration durationOfStorage;
	private final TimeZone utc;
	private final DateFormat dfFolder;
	private final DateFormat dfDoyFolder;
	private final DateFormat dfCycle;
	
	public DateUtils(int datePosition, Sensor.Duration durationOfStorage) {
		this.datePosition = datePosition;
		this.durationOfStorage = durationOfStorage;
		this.utc = TimeZone.getTimeZone("GMT");
		this.dfFolder = new SimpleDateFormat("yyyy/MM/dd");
		this.dfDoyFolder = new SimpleDateFormat("yyyy/DDD");
		this.dfCycle = new SimpleDateFormat("yyyyMMddHHmm");
		dfFolder.setTimeZone(utc);
		dfDoyFolder.setTimeZone(utc);
		dfCycle.setTimeZone(utc);
	}
	
	public boolean fileIsOld(File file) {
		int allowedAgeInDays;
		switch (durationOfStorage) {
		case FOREVER:
			return false;
		case DELETE_IMMEDIATELY:
			return true;
		case ONE_DAY:
			allowedAgeInDays = 1;
			break;
		case TWO_DAYS:
			allowedAgeInDays = 2;
			break;
		case ONE_WEEK:
			allowedAgeInDays = 7;
			break;
		case ONE_MONTH:
			allowedAgeInDays = 31;
			break;
		case SIX_MONTHS:
			allowedAgeInDays = 184;
			break;
		case ONE_YEAR:
			allowedAgeInDays = 366;
			break;
		default:
			allowedAgeInDays = -1;
			break;
		}
		
		if (allowedAgeInDays >= 0) {
			Calendar fileDate = getFileDate(file);
			Calendar expirationDate = getExpirationDate(allowedAgeInDays);
			return fileDate.before(expirationDate);
			
		} else
			return false;
		
	}
	
	public Calendar getFileDateTime(File file) {
		Calendar calendar;
		try {
			String fileName = file.getName();
			int timeOffset = separateDateTime(fileName)?1:0;
			String year = fileName.substring(datePosition, datePosition + 4);
			String month = fileName.substring(datePosition + 4, datePosition + 6);
			String day = fileName.substring(datePosition + 6, datePosition + 8);
			String hours = fileName.substring(datePosition + 8 + timeOffset, datePosition + 10 + timeOffset);
			String mins = fileName.substring(datePosition + 10 + timeOffset, datePosition + 12 + timeOffset);
			calendar = new GregorianCalendar(Integer.parseInt(year), Integer.parseInt(month) - 1, Integer.parseInt(day), Integer.parseInt(hours), Integer.parseInt(mins));
			calendar.setTimeZone(utc);
		} catch (Exception e) {
			calendar = new GregorianCalendar();
			calendar.setTimeInMillis(file.lastModified());
		}
		return calendar;
	}
	
	private boolean separateDateTime(String fileName) {
		char c = fileName.charAt(datePosition + 8);
		boolean separateDateTime = false;
		if ((c < '0') || (c > '9'))
			separateDateTime = true;
		return separateDateTime;
	}

	
	private Calendar getFileDate(File file) {
		Calendar calendar;
		try {
			String fileName = file.getName();
			String year = fileName.substring(datePosition, datePosition + 4);
			String month = fileName.substring(datePosition + 4, datePosition + 6);
			String day = fileName.substring(datePosition + 6, datePosition + 8);
			calendar = new GregorianCalendar(Integer.parseInt(year), Integer.parseInt(month) - 1, Integer.parseInt(day));
			calendar.setTimeZone(utc);
		} catch (Exception e) {
			calendar = new GregorianCalendar();
			calendar.setTimeInMillis(file.lastModified());
		}
		return calendar;
	}

	private Calendar getFileDoyDate(File file) {
		Calendar calendar;
		try {
			String fileName = file.getName();
			String year = fileName.substring(datePosition, datePosition + 4);
			String doy = fileName.substring(datePosition + 4, datePosition + 7);
			calendar = new GregorianCalendar();
			calendar.set(Calendar.YEAR, Integer.parseInt(year));
			calendar.set(Calendar.DAY_OF_YEAR, Integer.parseInt(doy));
			calendar.setTimeZone(utc);
		} catch (Exception e) {
			calendar = new GregorianCalendar();
			calendar.setTimeInMillis(file.lastModified());
		}
		return calendar;
	}

	public static Calendar getExpirationDate(int allowedAgeInDays) {
		Calendar calendar = new GregorianCalendar();
		calendar.add(Calendar.DAY_OF_MONTH, -allowedAgeInDays);
		return calendar;
	}
	
	public int diffInMins(File file, String cycle) {
		return (int)((getFileDateTime(file).getTimeInMillis() - getCycleDate(cycle).getTimeInMillis()) / 60000);
	}
	
	private Calendar getCycleDate(String cycle) {
		Calendar calendar = new GregorianCalendar();
		calendar.setTimeZone(utc);
		try {
			calendar.setTime(dfCycle.parse(cycle));
		} catch (ParseException e) {
		}
		return calendar;
	}
	
	public String getTime(File file) {
		String fileName = file.getName();
		int timeOffset = separateDateTime(fileName)?1:0;
		String time = fileName.substring(datePosition + 8 + timeOffset, datePosition + 12 + timeOffset);
		return time;
	}

	public String getDatedFolder(File file) {
		return dfFolder.format(getFileDate(file).getTime()); 
	}

	public String getDoyFolder(File file) {
		return dfDoyFolder.format(getFileDoyDate(file).getTime()); 
	}

	public String getCycle(File file) {
		return dfCycle.format(getFileDateTime(file).getTime()); 
	}
}
