/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.map.proj;

import gov.nasa.giss.graphics.Bezier;
import gov.nasa.giss.graphics.GraphicUtils;
import gov.nasa.giss.map.MapUtils;
import gov.nasa.giss.map.proj.AbstractProjection;
import gov.nasa.giss.map.proj.ProjectionUtils;
import gov.nasa.giss.math.PointLL;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WilliamOlsson
extends AbstractProjection {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String PROJECTION_NAME = "William-Olsson";
    public static final int PROPERTIES = 16793602;
    private static final double JOIN_LAT = 20.0;
    private static final double JOIN_LAT_RAD = Math.toRadians(20.0);
    private static final double JOIN_COLAT_RAD = 1.5707963267948966 - JOIN_LAT_RAD;
    private static final double COS_JOIN = Math.cos(JOIN_LAT_RAD);
    private static final double SIN_JOIN = Math.sin(JOIN_LAT_RAD);
    private static final double SEC_JOIN = 1.0 / COS_JOIN;
    private static final double K_JOIN = Math.sqrt(2.0 / (1.0 + SIN_JOIN));
    private static final double RHO_JOIN = K_JOIN * COS_JOIN;
    private static final double COS_HALF_J_COLAT = Math.cos(0.5 * JOIN_COLAT_RAD);
    private static final int NUM_LOBES = 4;
    private static final double LOBE_WIDTH = 90.0;
    private static final double HALF_LOBE_WIDTH = 45.0;
    private static final double LOBE_WIDTH_RAD = Math.toRadians(90.0);
    private static final double HALF_LOBE_WIDTH_RAD = 0.5 * LOBE_WIDTH_RAD;
    private static final double COS_HALF_LOBE_WIDTH = Math.cos(HALF_LOBE_WIDTH_RAD);
    private static final double SIN_HALF_LOBE_WIDTH = Math.sin(HALF_LOBE_WIDTH_RAD);
    private static final double COS_LOBE_WIDTH = Math.cos(LOBE_WIDTH_RAD);
    private static final double SIN_LOBE_WIDTH = Math.sin(LOBE_WIDTH_RAD);
    private static final double LOBE_SCALING = RHO_JOIN / JOIN_COLAT_RAD;
    private static final double INV_LOBE_SCALING = 1.0 / LOBE_SCALING;
    private static final double MER_SCALING = COS_HALF_J_COLAT * SEC_JOIN;
    private static final double INV_MER_SCALING = 1.0 / MER_SCALING;
    private static final double PAR_SCALING = JOIN_COLAT_RAD * SEC_JOIN;
    private static final double INV_PAR_SCALING = 1.0 / PAR_SCALING;
    private static final double SIZE_FACTOR;
    private static final double REFERENCE_LON = -20.0;

    public WilliamOlsson(int width, int height) {
        this(width, height, 0, 0);
    }

    public WilliamOlsson(Dimension size, Dimension margin) {
        this(size.width, size.height, margin.width, margin.height);
    }

    public WilliamOlsson(int width, int height, int xmargin, int ymargin) {
        super(PROJECTION_NAME, 16793602, width, height, xmargin, ymargin, SIZE_FACTOR, SIZE_FACTOR);
        super.setCenter(-20.0, 90.0);
    }

    @Override
    public void setCenter(double lon, double lat) {
        LOGGER.trace("Projection does not support changing center.");
    }

    @Override
    public boolean isRecenterableLon() {
        return false;
    }

    @Override
    public boolean isRecenterableLat() {
        return false;
    }

    @Override
    public Point2D.Double transformLL2XYIgnoreMargins(double lon, double lat) {
        double y;
        double x;
        if (Math.abs(lat) > 90.0) {
            throw new IllegalArgumentException("Latitude must be in range [-90,90]&#176;.");
        }
        if (lat > 89.99999) {
            return new Point2D.Double(this.outCenterX_, this.outCenterY_);
        }
        double phiRad = Math.toRadians(lat);
        if (lat >= 20.0) {
            double halfPhiRad = 0.5 * phiRad;
            double cosHalfPhi = Math.cos(halfPhiRad);
            double sinHalfPhi = Math.sin(halfPhiRad);
            double rho = 1.4142135623730951 * (cosHalfPhi - sinHalfPhi);
            double lambdaRad = this.lonToLambdaRad(lon);
            x = rho * Math.sin(lambdaRad);
            y = -rho * Math.cos(lambdaRad);
        } else {
            double betaRad;
            double alphaRad = JOIN_COLAT_RAD + MER_SCALING * (JOIN_LAT_RAD - phiRad);
            double rho = LOBE_SCALING * alphaRad;
            double lambda = MapUtils.normalize360(this.lonToLambda(lon));
            int lobe = (int)(lambda / 90.0);
            if (lat < -89.99999) {
                betaRad = 0.0;
            } else {
                double xlambda = lambda - 90.0 * (double)lobe - 45.0;
                double xlambdaRad = Math.toRadians(xlambda);
                betaRad = PAR_SCALING * xlambdaRad * Math.cos(phiRad) / alphaRad;
            }
            double xx = rho * Math.sin(betaRad);
            double yy = -rho * Math.cos(betaRad);
            double xxx = xx * COS_HALF_LOBE_WIDTH - yy * SIN_HALF_LOBE_WIDTH;
            double yyy = xx * SIN_HALF_LOBE_WIDTH + yy * COS_HALF_LOBE_WIDTH;
            for (int i = 1; i <= lobe; ++i) {
                x = COS_LOBE_WIDTH * xxx - SIN_LOBE_WIDTH * yyy;
                y = SIN_LOBE_WIDTH * xxx + COS_LOBE_WIDTH * yyy;
                xxx = x;
                yyy = y;
            }
            x = xxx;
            y = yyy;
        }
        x = (double)this.outCenterX_ + x * this.rS_;
        y = (double)this.outCenterY_ - y * this.rS_;
        return new Point2D.Double(x, y);
    }

    @Override
    public PointLL transformXY2LL(double xx, double yy) {
        double lambdaRad;
        double phiRad;
        double x = xx - (double)this.outCenterX_;
        double y = (double)this.outCenterY_ - yy;
        double absx = Math.abs(x);
        double absy = Math.abs(y);
        if (absx > (double)this.dxMax_ || absy > (double)this.dyMax_) {
            return null;
        }
        if (x == 0.0 && y == 0.0) {
            return this.getCenter();
        }
        double rho = Math.sqrt(x * x + y * y) * this.invRS_;
        if (rho <= RHO_JOIN) {
            double z = 2.0 * Math.asin(0.5 * rho);
            double sinZ = Math.sin(z);
            phiRad = 1.5707963267948966 - z;
            lambdaRad = Math.atan2(x * sinZ, -y * sinZ);
        } else {
            double alphaRad = rho * INV_LOBE_SCALING;
            phiRad = JOIN_LAT_RAD - (alphaRad - JOIN_COLAT_RAD) * INV_MER_SCALING;
            if (phiRad < -1.5707963267948966) {
                return null;
            }
            double cosPhi = Math.cos(phiRad);
            int lobe = 0;
            if (x > 0.0 && y > 0.0) {
                lobe = 1;
            } else if (x < 0.0) {
                lobe = y > 0.0 ? 2 : 3;
            }
            double betaRad = Math.atan2(absx, absy);
            lambdaRad = alphaRad * (betaRad - HALF_LOBE_WIDTH_RAD) * INV_PAR_SCALING / cosPhi;
            lambdaRad += HALF_LOBE_WIDTH_RAD;
            if (lambdaRad > 1.5707963267948966) {
                return null;
            }
            for (lambdaRad = MapUtils.normalizeTwoPi(lambdaRad); lambdaRad < 0.0; lambdaRad += Math.PI * 2) {
            }
            switch (lobe) {
                case 1: {
                    lambdaRad = Math.PI - lambdaRad;
                    break;
                }
                case 2: {
                    lambdaRad += Math.PI;
                    break;
                }
                case 3: {
                    lambdaRad = Math.PI * 2 - lambdaRad;
                    break;
                }
            }
        }
        double phi = Math.toDegrees(phiRad);
        if (phi < -90.0) {
            return null;
        }
        double lambda = Math.toDegrees(lambdaRad);
        return new PointLL(this.lambdaC_ + lambda, phi);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void calculateInverseArray() {
        WilliamOlsson williamOlsson = this;
        synchronized (williamOlsson) {
            for (int iiy = 0; iiy < this.dyMax_; ++iiy) {
                int iy = -iiy;
                double y = (double)iy + 0.5;
                double yOverRS = y * this.invRS_;
                double y2OverRS2 = yOverRS * yOverRS;
                for (int ix = 0; ix < this.dxMax_; ++ix) {
                    double lambdaRad;
                    double phiRad;
                    double x = (double)ix + 0.5;
                    double xOverRS = x * this.invRS_;
                    double rho = Math.sqrt(xOverRS * xOverRS + y2OverRS2);
                    if (rho <= RHO_JOIN) {
                        double z = 2.0 * Math.asin(0.5 * rho);
                        double sinZ = Math.sin(z);
                        phiRad = 1.5707963267948966 - z;
                        lambdaRad = Math.atan2(x * sinZ, -y * sinZ);
                    } else {
                        double alphaRad = rho * INV_LOBE_SCALING;
                        phiRad = JOIN_LAT_RAD - (alphaRad - JOIN_COLAT_RAD) * INV_MER_SCALING;
                        if (phiRad < -1.5707963267948966) continue;
                        double betaRad = Math.atan2(x, -y);
                        lambdaRad = alphaRad * (betaRad - HALF_LOBE_WIDTH_RAD) * INV_PAR_SCALING / Math.cos(phiRad);
                        lambdaRad += HALF_LOBE_WIDTH_RAD;
                        if ((lambdaRad = MapUtils.normalizeTwoPi(lambdaRad)) > 1.5707963267948966) continue;
                    }
                    double lambda = Math.toDegrees(lambdaRad);
                    double phi = Math.toDegrees(phiRad);
                    this.set4SymmetricPoints(ix, iy, lambda, phi);
                }
            }
        }
    }

    private void set4SymmetricPoints(int ix, int iy, double dlambda, double phi) {
        int col = this.outCenterX_ + ix;
        int colF = this.outCenterX_ - 1 - ix;
        int row = this.outCenterY_ - 1 - iy;
        int rowF = this.outCenterY_ + iy;
        if (row < 0 || row >= this.outHeight_ || rowF < 0 || rowF >= this.outHeight_ || col < 0 || col >= this.outWidth_ || colF < 0 || colF >= this.outWidth_) {
            return;
        }
        double lonTR = this.lambdaC_ + dlambda;
        double lonTL = this.lambdaC_ - dlambda;
        int indexTR = row * this.outWidth_ + col;
        int indexTL = row * this.outWidth_ + colF;
        int indexBL = rowF * this.outWidth_ + colF;
        int indexBR = rowF * this.outWidth_ + col;
        this.invArrayLon_[indexTR] = lonTR;
        this.invArrayLon_[indexTL] = lonTL;
        this.invArrayLon_[indexBL] = lonTR + 180.0;
        this.invArrayLon_[indexBR] = lonTL + 180.0;
        this.invArrayLat_[indexTR] = phi;
        this.invArrayLat_[indexTL] = phi;
        this.invArrayLat_[indexBL] = phi;
        this.invArrayLat_[indexBR] = phi;
        int srcY = this.getSrcPixelY(phi);
        if (srcY < 0) {
            return;
        }
        int srcXTR = this.getSrcPixelX(this.invArrayLon_[indexTR]);
        int srcXTL = this.getSrcPixelX(this.invArrayLon_[indexTL]);
        int srcXBL = this.getSrcPixelX(this.invArrayLon_[indexBL]);
        int srcXBR = this.getSrcPixelX(this.invArrayLon_[indexBR]);
        if (srcXTR > -1) {
            this.invArray_[indexTR] = srcY * this.srcWidth_ + srcXTR;
        }
        if (srcXTL > -1) {
            this.invArray_[indexTL] = srcY * this.srcWidth_ + srcXTL;
        }
        if (srcXBL > -1) {
            this.invArray_[indexBL] = srcY * this.srcWidth_ + srcXBL;
        }
        if (srcXBR > -1) {
            this.invArray_[indexBR] = srcY * this.srcWidth_ + srcXBR;
        }
    }

    @Override
    protected void drawBorderLines(Graphics2D g2d) {
        for (int i = 0; i < 4; ++i) {
            double lon1 = this.lambdaC_ + (double)i * 90.0 + 1.0E-5;
            double lon2 = this.lambdaC_ + (double)(i + 1) * 90.0 - 1.0E-5;
            Bezier bcurve = this.makeMeridianBezier(lon1);
            if (bcurve != null) {
                bcurve.draw(g2d);
            }
            if ((bcurve = this.makeMeridianBezier(lon2)) == null) continue;
            bcurve.draw(g2d);
        }
    }

    @Override
    protected void drawParallel(Graphics2D g2d, double lat, String label) {
        if (lat >= 20.0) {
            Point2D.Double dot = this.transformLL2XYIgnoreMargins(this.lambdaC_ + 45.0, lat);
            if (dot == null) {
                return;
            }
            double dx = (double)this.outCenterX_ - dot.x;
            double dy = (double)this.outCenterY_ - dot.y;
            double r = Math.sqrt(dx * dx + dy * dy);
            g2d.translate(this.outCenterX_, this.outCenterY_);
            ProjectionUtils.drawEllipse(g2d, 0.0, 0.0, r, r);
            g2d.translate(-this.outCenterX_, -this.outCenterY_);
        } else {
            for (int i = 0; i < 4; ++i) {
                double lon1 = this.lambdaC_ + (double)i * 90.0 + 1.0E-5;
                double lon2 = this.lambdaC_ + (double)i * 90.0 + 45.0;
                double lon3 = this.lambdaC_ + (double)(i + 1) * 90.0 - 1.0E-5;
                Point2D.Double dot1 = this.transformLL2XYIgnoreMargins(lon1, lat);
                Point2D.Double dot2 = this.transformLL2XYIgnoreMargins(lon2, lat);
                Point2D.Double dot3 = this.transformLL2XYIgnoreMargins(lon3, lat);
                GraphicUtils.drawCircularArc(g2d, dot1.x, dot1.y, dot2.x, dot2.y, dot3.x, dot3.y);
            }
        }
    }

    @Override
    protected void drawMeridian(Graphics2D g2d, double lon, String label) {
        for (int i = 0; i < 4; ++i) {
            double lobeCenter = MapUtils.normalizeMP180(this.lambdaC_ + (double)i * 90.0 + 45.0);
            double cdiff = Math.abs(MapUtils.normalizeMP180(lon - lobeCenter));
            if (cdiff > 45.0) continue;
            Point2D.Double dot90S = this.transformLL2XYIgnoreMargins(lon, -90.0);
            if (Math.abs(cdiff) < 1.0E-5) {
                GraphicUtils.drawLine(g2d, this.outCenterX_, this.outCenterY_, dot90S.x, dot90S.y);
                return;
            }
            Point2D.Double dotEq = this.transformLL2XYIgnoreMargins(lon, 20.0);
            GraphicUtils.drawLine(g2d, this.outCenterX_, this.outCenterY_, dotEq.x, dotEq.y);
            if (cdiff > 44.99999) {
                return;
            }
            Bezier bcurve = this.makeMeridianBezier(lon);
            if (bcurve == null) continue;
            bcurve.draw(g2d);
        }
    }

    private Bezier makeMeridianBezier(double lon) {
        Point2D.Double dot;
        ArrayList<Point2D.Double> ptlist = new ArrayList<Point2D.Double>(240);
        for (double lat = -90.0; lat < 20.0; lat += 0.5) {
            dot = this.transformLL2XYIgnoreMargins(lon, lat);
            ptlist.add(new Point2D.Double(dot.x, dot.y));
        }
        dot = this.transformLL2XYIgnoreMargins(lon, 20.0);
        ptlist.add(new Point2D.Double(dot.x, dot.y));
        return new Bezier(false, ptlist);
    }

    static {
        double maxLat = -74.388;
        double maxLatRad = Math.toRadians(-74.388);
        double alphaRad = JOIN_COLAT_RAD + MER_SCALING * (JOIN_LAT_RAD - maxLatRad);
        double rho = LOBE_SCALING * alphaRad;
        double betaRad = -PAR_SCALING * HALF_LOBE_WIDTH_RAD * Math.cos(maxLatRad) / alphaRad;
        SIZE_FACTOR = rho * Math.abs((Math.sin(betaRad) - Math.cos(betaRad)) * COS_HALF_LOBE_WIDTH);
    }
}

