import java.awt.Container;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Pattern;

import javax.swing.JPanel;

public class Channel {
	private String name;
	private String parentName;
	private String filePattern;
	private Pattern fileRegex;
	private Value processChannel;
	private int nrOfSegments;
	private Value segmentsToStore;
	private int timesPerDay;
	private int minutesBetweenCycles;
	private Value timesToStore;

	private List<File> filesToProcess;
	private List<File> filesToDelete;
	private List<String> times;
	private List<Integer> segmentNrs;

	private String currentCycle;
	private int countSegments;

	private static final String CHANNEL_NAME = "Item Name:";
	private static final String CHANNEL_PATTERN = "Pattern:";
	private static final String SEGMENTS_PER_CHANNEL = "Expected segments:";
	private static final String PROCESS_CHANNEL = "Process:";
	private static final String SEGMENTS_TO_STORE = "Segments to store:";
	private static final String FRAMES_PER_DAY = "Times per day:";
	private static final String FRAMES_TO_STORE = "Times to store:";

	public Channel(String parentName) {
		this.parentName = parentName;
		processChannel = new Value(false);
		timesToStore = new Value("all");
		segmentsToStore = new Value("all");
		currentCycle = "";
		countSegments = 0;
	}

	private static String wildcardToRegex(String wildcard) {
		StringBuffer s = new StringBuffer(wildcard.length());
		s.append('^');
		for (int i = 0, is = wildcard.length(); i < is; i++) {
			char c = wildcard.charAt(i);
			switch (c) {
			case '*':
				s.append(".*");
				break;
			case '?':
				s.append(".");
				break;
			// escape special regexp-characters
			case '(':
			case ')':
			case '[':
			case ']':
			case '$':
			case '^':
			case '.':
			case '{':
			case '}':
			case '|':
			case '\\':
				s.append("\\");
				s.append(c);
				break;
			default:
				s.append(c);
				break;
			}
		}
		s.append('$');
		return (s.toString());
	}

	public boolean read(ConfigFile configFile) {
		this.name = configFile.readToken(CHANNEL_NAME, "no_more_channels");
		if (name.compareTo("no_more_channels") != 0) {
			this.filePattern = configFile.readToken(CHANNEL_PATTERN, "");
			String regex = wildcardToRegex(filePattern);
			this.fileRegex = Pattern.compile(regex);
			this.processChannel.setVal(configFile.readToken(PROCESS_CHANNEL, "yes").compareToIgnoreCase("yes") == 0);
			this.timesPerDay = Integer.parseInt(configFile.readToken(FRAMES_PER_DAY, "0"));
			this.timesToStore.setVal(configFile.readToken(FRAMES_TO_STORE, "all"));
			this.nrOfSegments = Integer.parseInt(configFile.readToken(SEGMENTS_PER_CHANNEL, "0"));
			this.segmentsToStore.setVal(configFile.readToken(SEGMENTS_TO_STORE, "all"));
			this.minutesBetweenCycles = (timesPerDay > 0) ? (1440 / timesPerDay) : 0;
			return true;
		} else
			return false;
	}

	public void write(ConfigFile configFile) {
		configFile.writeToken(CHANNEL_NAME, name, null);
		configFile.writeToken(CHANNEL_PATTERN, filePattern, "");
		configFile.writeToken(PROCESS_CHANNEL, processChannel.getBoolVal() ? "yes" : "no", "yes");
		configFile.writeToken(FRAMES_PER_DAY, "" + timesPerDay, "0");
		configFile.writeToken(FRAMES_TO_STORE, timesToStore.getStringVal(), "all");
		configFile.writeToken(SEGMENTS_PER_CHANNEL, "" + nrOfSegments, "0");
		configFile.writeToken(SEGMENTS_TO_STORE, segmentsToStore.getStringVal(), "all");
		configFile.writeln();
	}

	public void addToPanel(Container panel, FormUtils fu, int xpos, int ypos) {
		JPanel channelPanel = fu.createGridPanel(name);
		int y = 0;
		fu.addCheckBox(channelPanel, PROCESS_CHANNEL, processChannel, 0, y++);
		// fu.addTextBox(channelPanel, CHANNEL_PATTERN, pattern, 0, y++);
		if (timesPerDay > 1) {
			// fu.addTextBox(channelPanel, FRAMES_PER_DAY, timesPerDay, false, 0, y++);
			fu.addTextBox(channelPanel, FRAMES_TO_STORE, "Which of the " + timesPerDay + " " + FRAMES_PER_DAY.substring(0, FRAMES_PER_DAY.length() - 1).toLowerCase() + " to store", timesToStore,
					true, 0, y++);
		}
		if (nrOfSegments > 1) {
			// fu.addTextBox(channelPanel, SEGMENTS_PER_CHANNEL, nrOfSegments, false, 0, y++);
			fu.addTextBox(channelPanel, SEGMENTS_TO_STORE, "Which of the " + nrOfSegments + " " + SEGMENTS_PER_CHANNEL.substring(0, SEGMENTS_PER_CHANNEL.length() - 1).toLowerCase() + " to store",
					segmentsToStore, true, 0, y++);
		}
		fu.addPanel(panel, channelPanel, xpos, ypos);
	}

	public void initFileProcessing() {
		filesToProcess = new ArrayList<File>();
		filesToDelete = new ArrayList<File>();

		if (timesToStore.getStringVal().compareToIgnoreCase("all") == 0) {
			times = null;
		} else {
			times = new ArrayList<String>();
			String[] tokens = timesToStore.getStringVal().split("\\s*,\\s*");
			if (tokens != null) {
				for (int i = 0; i < tokens.length; ++i) {
					times.add(tokens[i]);
				}
			}
		}

		if (segmentsToStore.getStringVal().compareToIgnoreCase("all") == 0) {
			segmentNrs = null;
		} else {
			segmentNrs = new ArrayList<Integer>();
			String[] tokens = segmentsToStore.getStringVal().split("\\s*,\\s*");
			if (tokens != null) {
				for (int i = 0; i < tokens.length; ++i) {
					int segmentNr = Integer.parseInt(tokens[i]);
					segmentNrs.add(segmentNr);
				}
			}
		}
	}

	private int getSegmentNr(File file, int fileIdPosition) {
		return Integer.parseInt(file.getName().substring(fileIdPosition, fileIdPosition + 6));
	}

	public boolean acceptFile(File file, DateUtils du, int fileIdPosition) {
		if (isMyFile(file)) {
			if (((times != null) && !times.contains(du.getTime(file))) || du.fileIsOld(file) || ((segmentNrs != null) && !segmentNrs.contains(getSegmentNr(file, fileIdPosition))))
				filesToDelete.add(file);
			else
				filesToProcess.add(file);

			return true;
		} else
			return false;
	}

	public boolean isMyFile(File file) { // are we "responsible" for this file?
		if (processChannel.getBoolVal())
			return fileRegex.matcher(file.getName()).matches();
		else
			return false;

		// if (processChannel.getBoolVal() && file.getName().startsWith(filePrefix))
		// return true;
		// else
		// return false;
	}

	public void processFiles(final String dataFolder, final boolean copy, final DateUtils du, final Sensor.FolderStorage datedFolders, final CycleMonitor cycleMonitor, final ActivityMonitor activityMonitor,
			final Logfile log, final LogWindow logWindow) {
		times = null;
		segmentNrs = null;

		activityMonitor.startJob(parentName + ":" + name + ": Processing ...", filesToProcess.size() + filesToDelete.size());

		List<File> filesToReport = new ArrayList<File>();
		filesToReport.addAll(filesToProcess);
		filesToReport.addAll(filesToDelete);

		Collections.sort(filesToReport, new Comparator<File>() {
			public int compare(File f1, File f2) {
				int cmp = du.getFileDateTime(f1).compareTo(du.getFileDateTime(f2));
				if (cmp != 0)
					return cmp;
				else
					return f1.compareTo(f2);
			}
		});

		for (File file : filesToReport) {
			if (minutesBetweenCycles > 0)
				logMissing(file, du, minutesBetweenCycles, log, logWindow);
			cycleMonitor.reportFile(file, du);
		}

		if (!copy) {
			for (File file : filesToDelete) {
				boolean success = file.delete();
				if (!success)
					logWindow.log("Could not delete " + file.getAbsolutePath(), true);
				activityMonitor.work();
			}
		}

		filesToDelete = null;

		File destinationDir;
		for (File file : filesToProcess) {
			String fileName = file.getName();
			if (datedFolders == Sensor.FolderStorage.yes)
				destinationDir = new File(dataFolder, du.getDatedFolder(file));
			else if (datedFolders == Sensor.FolderStorage.doy)
				destinationDir = new File(dataFolder, du.getDoyFolder(file));
			else
				destinationDir = new File(dataFolder);
			destinationDir.mkdirs();
			File destinationFile = new File(destinationDir, fileName);
			if (copy) {
				if (!destinationFile.exists()) {
					boolean success = FileProcessor.copyFile(file, destinationFile);
					if (!success) {
						logWindow.log("Could not copy " + file.getAbsolutePath() + " to " + destinationDir, true);
						destinationFile.delete();
					}
				}
			} else {
				if (destinationFile.exists())
					destinationFile.delete();
				boolean success = file.renameTo(destinationFile);
				if (!success)
					logWindow.log("Could not move " + file.getAbsolutePath() + " to " + destinationDir, true);
			}
			activityMonitor.work();
		}

		activityMonitor.endJob();

		filesToProcess = null;
	}

	private void logMissing(File file, DateUtils du, int minutesBetweenCycles, Logfile log, LogWindow logWindow) {
		String cycle = du.getCycle(file);
		if (currentCycle != null && currentCycle.length() > 0) {
			if (cycle.compareTo(currentCycle) > 0) {
				// check if we have an indication that something is missing, and reset counters
				if (countSegments < nrOfSegments)
					log.log("Missing " + (nrOfSegments - countSegments) + " segment(s) for " + parentName + ":" + name + " at cycle " + currentCycle, logWindow);
				countSegments = 0;
				if (minutesBetweenCycles > 0) {
					int missingCycles = du.diffInMins(file, currentCycle) / minutesBetweenCycles - 1;
					if (missingCycles > 0)
						log.log("Missing " + missingCycles + " cycle(s) for " + parentName + ":" + name + " after cycle " + currentCycle, logWindow);
				}

				// commit cycle increase
				currentCycle = cycle;
			}
		} else {// first time
			currentCycle = cycle;
		}
		++countSegments;
	}

	public int nrExpectedFilesPerCycle() {
		return nrOfSegments;
	}

	public String checkData() {
		String violatingItem = parentName + ":" + name;

		// CHECK segmentsToStore
		if (segmentsToStore.getStringVal().length() == 0)
			return violatingItem + ": Please enter at least one segment to store, or 'all'";
		if (segmentsToStore.getStringVal().compareToIgnoreCase("all") != 0) {
			try {
				String[] tokens = segmentsToStore.getStringVal().split("\\s*,\\s*");
				if (tokens != null && tokens.length > 0) {
					for (int i = 0; i < tokens.length; ++i) {
						int segmentNr = Integer.parseInt(tokens[i]);
						if (segmentNr < 1)
							return violatingItem + ": The minimum segment number possible is 1.";
						else if (segmentNr > nrOfSegments)
							return violatingItem + ": The maximum segment number possible is " + nrOfSegments + ".";
					}
				} else
					return violatingItem + ": Failed to understand the segments to store. Please enter 'all' or a comma-separated list of the segment numbers to store.";
			} catch (Exception e) {
				return violatingItem + ": Failed to understand the segments to store. Please enter 'all' or a comma-separated list of the segment numbers to store.";
			}
		}

		// CHECK timesToStore
		if (timesToStore.getStringVal().length() == 0)
			return violatingItem + ": Please enter at least one time to store, or 'all'";
		if (timesToStore.getStringVal().compareToIgnoreCase("all") != 0) {
			try {
				String[] tokens = timesToStore.getStringVal().split("\\s*,\\s*");
				if (tokens != null && tokens.length > 0) {
					for (int i = 0; i < tokens.length; ++i) {
						if (tokens[i].length() != 4)
							return violatingItem + ": Failed to understand the times to store. Please enter 'all' or a comma-separated list of four-digit times to store (e.g. 1300,1315,1330).";
						int hours = Integer.parseInt(tokens[i].substring(0, 2));
						int minutes = Integer.parseInt(tokens[i].substring(2));
						if (hours < 0 || hours > 23)
							return violatingItem + ": The hours can be between 00 and 23.";
						else if (minutes < 0 || minutes > 59)
							return violatingItem + ": The minutes can be between 00 and 59.";
					}
				} else
					return violatingItem + ": Failed to understand the times to store. Please enter 'all' or a comma-separated list of four-digit times to store (e.g. 1300,1315,1330).";
			} catch (Exception e) {
				return violatingItem + ": Failed to understand the times to store. Please enter 'all' or a comma-separated list of four-digit times to store (e.g. 1300,1315,1330).";
			}
		}

		// All OK
		return "";
	}
}
