import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;

public class Sensor {

	public enum Duration {
		ONE_DAY, TWO_DAYS, ONE_WEEK, ONE_MONTH, SIX_MONTHS, ONE_YEAR, FOREVER, DELETE_IMMEDIATELY
	};
	
	public enum FolderStorage {
		no, yes, doy
	};

	private String name;
	private String description;
	private int datePosition;
	private int fileIdPosition;
	private Value dataFolder;
	private Value datedFolders;
	private Value missingDataFile;
	private Value durationOfStorage;

	private List<Channel> channels;

	private DateUtils du;
	private CycleMonitor cycleMonitor;

	private static final String NAME = "Group Name:";
	private static final String DESCRIPTION = "Description:";
	private static final String DATE_POSITION = "Date position:";
	private static final String FILEID_POSITION = "File id position:";
	private static final String DATA_FOLDER = "Destination folder:";
	private static final String DATED_FOLDERS = "Dated folders:";
	private static final String MISSING_DATA_FILE = "Missing data log:";
	private static final String DURATION_OF_STORAGE = "Duration of storage:";

	public Sensor() {
		channels = new ArrayList<Channel>();
		dataFolder = new Value("");
		datedFolders = new Value(FolderStorage.no);
		missingDataFile = new Value("");
		durationOfStorage = new Value(Duration.FOREVER);
	}

	public int nrExpectedFilesPerCycle() {
		int sum = 0;
		for (Channel channel : channels)
			sum += channel.nrExpectedFilesPerCycle();
		return sum;
	}

	public String getName() {
		return name;
	}

	public boolean read(ConfigFile configFile) {
		this.name = configFile.readToken(NAME, "no_more_sensors");
		if (name.compareTo("no_more_sensors") != 0) {
			this.description = configFile.readToken(DESCRIPTION, name);
			this.datePosition = Integer.parseInt(configFile.readToken(DATE_POSITION, "46"));
			this.fileIdPosition = Integer.parseInt(configFile.readToken(FILEID_POSITION, "36"));
			this.dataFolder.setVal(configFile.readToken(DATA_FOLDER, ""));
			this.datedFolders.setVal(FolderStorage.valueOf(configFile.readToken(DATED_FOLDERS, FolderStorage.yes.toString())));
			this.missingDataFile.setVal(configFile.readToken(MISSING_DATA_FILE, ""));
			this.durationOfStorage.setVal(Duration.valueOf(configFile.readToken(DURATION_OF_STORAGE, Duration.FOREVER.toString())));

			Channel channel = new Channel(name);
			while (channel.read(configFile)) {
				channels.add(channel);
				channel = new Channel(name);
			}
			return true;
		} else
			return false;
	}

	public void write(ConfigFile configFile) {
		configFile.writeToken(NAME, name, null);
		configFile.writeToken(DESCRIPTION, description, name);
		configFile.writeToken(DATE_POSITION, "" + datePosition, "46");
		configFile.writeToken(FILEID_POSITION, "" + fileIdPosition, "36");

		configFile.writeToken(DATA_FOLDER, dataFolder.getStringVal(), "");
		configFile.writeToken(DATED_FOLDERS, datedFolders.getObjectVal().toString(), FolderStorage.yes.toString());
		configFile.writeToken(MISSING_DATA_FILE, missingDataFile.getStringVal(), "");
		configFile.writeToken(DURATION_OF_STORAGE, durationOfStorage.getObjectVal().toString(), Duration.FOREVER.toString());

		configFile.writeln();
		for (Channel channel : channels)
			channel.write(configFile);
	}

	public void addToPanel(Container panel, FormUtils fu, int cols) {
		JPanel sensorPanel = fu.createGridPanel(description);

		JPanel inputPanel = new JPanel();
		inputPanel.setLayout(new GridBagLayout());
		int ypos = 0;
		fu.addBrowseFolder(inputPanel, DATA_FOLDER, dataFolder, 0, ypos);
		JPanel datedFoldersCombo = new JPanel();
		datedFoldersCombo.setLayout(new GridBagLayout());
		JComboBox combo = fu.addComboBox(datedFoldersCombo, DATED_FOLDERS, datedFolders, FolderStorage.values(), 0, 0);
		combo.setMinimumSize(new Dimension(50,20));
		combo.setPreferredSize(new Dimension(50,20));
		fu.addPanel(inputPanel, datedFoldersCombo, 3, ypos++);
		fu.addBrowseFile(inputPanel, MISSING_DATA_FILE, missingDataFile, 0, ypos++);
		fu.addComboBox(inputPanel, DURATION_OF_STORAGE, durationOfStorage, Duration.values(), 0, ypos++);
		fu.addPanel(sensorPanel, inputPanel, 0, 0);

		JPanel progressPanel = new JPanel();
		progressPanel.setLayout(new GridBagLayout());

		int nrExpectedFilesPerCycle = nrExpectedFilesPerCycle();
		JTextField currentCycleText = fu.addTextBox(progressPanel, (nrExpectedFilesPerCycle > 1) ? "Current cycle:" : "Last cycle:", "", new Value(""), false, 0, 0);
		JProgressBar progressBar = (nrExpectedFilesPerCycle > 1) ? fu.addProgressBar(progressPanel, 1, 1) : null;
		fu.addPanel(sensorPanel, progressPanel, 1, 0);
		cycleMonitor = new CycleMonitor(progressBar, currentCycleText, nrExpectedFilesPerCycle());

		if (channels != null && channels.size() > 0) {
			JPanel channelPanel = fu.createGridPanel("Items");

			int channelNr = 0;
			for (Channel channel : channels) {
				int channelXpos = channelNr % cols;
				int channelYpos = channelNr / cols;
				channel.addToPanel(channelPanel, fu, channelXpos, channelYpos);
				++channelNr;
			}
			fu.addPanelEntireRow(sensorPanel, channelPanel, 0, 1);
		}
		panel.add(sensorPanel);
	}

	public void initFileProcessing() {
		du = new DateUtils(datePosition, (Duration) (durationOfStorage.getObjectVal()));
		for (Channel channel : channels)
			channel.initFileProcessing();
	}

	public boolean acceptFile(File file) {
		for (Channel channel : channels)
			if (channel.acceptFile(file, du, fileIdPosition))
				return true;
		return false;
	}

	public void processFiles(final boolean copy, final ActivityMonitor activityMonitor, final LogWindow logWindow) {
		Logfile log = new Logfile(missingDataFile.getStringVal());
		for (Channel channel : channels)
			channel.processFiles(dataFolder.getStringVal(), copy, du, (FolderStorage) datedFolders.getObjectVal(), cycleMonitor, activityMonitor, log, logWindow);
		du = null;
	}

	public File getRootFolder() {
		return new File(dataFolder.getStringVal());
	}

	public void deleteOldFiles(ActivityMonitor activityMonitor, List<File> rootFolders, LogWindow logWindow) {
		final DateUtils du = new DateUtils(datePosition, (Duration) (durationOfStorage.getObjectVal()));
		if (durationOfStorage.getObjectVal() != Duration.FOREVER) {
			activityMonitor.startJob(name + ": Deleting old files");
			File data = new File(dataFolder.getStringVal());
			deleteRecursively(new FilenameFilter() {
				public boolean accept(File parent, String fileName) {
					File file = new File(parent, fileName);
					for (Channel channel : channels)
						if (channel.isMyFile(file))
							return du.fileIsOld(file);
					return false;
				}
			}, new FilenameFilter() {
				public boolean accept(File parent, String fileName) {
					File file = new File(parent, fileName);
					if (file.isDirectory())
						return true;
					else
						return false;
				}
			}, data, rootFolders, logWindow);
			activityMonitor.endJob();
		}
	}

	private void deleteRecursively(FilenameFilter fileFilter, FilenameFilter folderFilter, File folder, List<File> rootFolders, LogWindow logWindow) {
		// Recurse first
		File[] folders = folder.listFiles(folderFilter);
		if (folders != null) {
			for (File f : folders) {
				deleteRecursively(fileFilter, folderFilter, f, rootFolders, logWindow);
			}
		} else
			logWindow.log("Could not read subfolders in " + folder.getAbsolutePath(), true);
		// Then delete own files
		File[] listFiles = folder.listFiles(fileFilter);
		if (listFiles != null && listFiles.length > 0) {
			for (File file : listFiles) {
				boolean success = file.delete();
				if (!success)
					logWindow.log("Could not delete " + file.getAbsolutePath(), true);
			}
		}
		// Then delete empty folders, but leave the root folders intact
		if (!rootFolders.contains(folder)) {
			listFiles = folder.listFiles();
			if (listFiles == null || listFiles.length == 0) {
				boolean success = folder.delete();
				if (!success)
					logWindow.log("Could not delete " + folder.getAbsolutePath(), true);
			}
		}
	}

	public List<Tuple> getDestinationFoldersToBeCreated() {
		List<Tuple> destinationFoldersToBeCreated = new ArrayList<Tuple>();
		File folder = new File(dataFolder.getStringVal());
		if (!folder.exists())
			destinationFoldersToBeCreated.add(new Tuple(folder, name));
		File file = new File(missingDataFile.getStringVal());
		if ((file.getPath().length() > 0) && (!file.isDirectory()) && (file.getParentFile() != null) && (!file.getParentFile().exists()))
			destinationFoldersToBeCreated.add(new Tuple(file.getParentFile(), name));
		return destinationFoldersToBeCreated;
	}

	public String checkData() {
		if (!new File(dataFolder.getStringVal()).exists())
			return name + ": Ensure that the destination folder exists.";

		File file = new File(missingDataFile.getStringVal());
		if ((file.getPath().length() == 0) || (file.isDirectory()))
			return name + ": Please enter a proper missing data log file.";
		else if (file.getParentFile() == null || !file.getParentFile().exists())
			return name + ": Please ensure the directory to place the missing data log file exists.";

		for (Channel channel : channels) {
			String msg = channel.checkData();
			if (msg.length() > 0)
				return msg;
		}

		// All OK
		return "";
	}
}
